%%%% A Matlab Code for 3D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT3D(100, 50, 20, 0.3, 4, 3)
function SEMDOT3D(nelx,nely,nelz, vol, rmin, ctp)
%% USER-DEFINED LOOP PARAMETERS
maxloop = 5000;
tolx = 0.001;
nele = nelx*nely*nelz;
%% USER-DEFINED MATERIAL PROPERTIES
E0 = 1; Emin = 0.001; nu = 0.3; tol = 1;
Obj_Scal=1; TopMax=[]; baseplate='S';
%% USER-DEFINED GRID POINTS
nG=5; ngrid=nG-1;
%% INITIALIZE HEAVISIDE REGULARIZATION PARAMETER
beta=0.5; ER=0.5;
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey,nodez] = meshgrid(0:nelx,0:nely,0:nelz);
[fnx,fny,fnz] = meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely,0:1/ngrid:nelz);
%% USER-DEFINED LOAD DOFs
switch(ctp)
    case 1 % A cantilever beam with distributed vertical load
        [il,jl,kl] = meshgrid(nelx, 0, 0:nelz);
        loadnid = kl*(nelx+1)*(nely+1)+il*(nely+1)+(nely+1-jl);
        loaddof = 3*loadnid(:) - 1;
        [iif,jf,kf] = meshgrid(0,0:nely,0:nelz);
        fixednid = kf*(nelx+1)*(nely+1)+iif*(nely+1)+(nely+1-jf);
        fixeddof = [3*fixednid(:); 3*fixednid(:)-1; 3*fixednid(:)-2];
    case 2 % A cantilever beam
        il = nelx; jl = 0; kl = nelz/2;
        loadnid = kl*(nelx+1)*(nely+1)+il*(nely+1)+(nely+1-jl);
        loaddof = 3*loadnid(:) - 1;
        [jf,kf] = meshgrid(1:nely+1,1:nelz+1);
        fixednid = (kf-1)*(nely+1)*(nelx+1)+jf;
        fixeddof = [3*fixednid(:); 3*fixednid(:)-1; 3*fixednid(:)-2];
    case 3 % A simply supported beam
        i1=nelx/2; j1=nely; k1=nelz/2;
        loadnid=k1*(nelx+1)*(nely+1)+i1*(nely+1)+(nely+1-j1);
        loaddof=3*loadnid(:)-1;
        iif=[0 0 nelx nelx]; jf=[0 0 0 0]; kf=[0 nelz 0 nelz];
        fixednid=kf*(nelx+1)*(nely+1)+iif*(nely+1)+(nely+1-jf);
        fixeddof=[3*fixednid(:);3*fixednid(:)-1;3*fixednid(:)-2];
    case 4 % A simply supported beam with distributed vertical load
        i1=nelx/2; j1=nely; k1=1:nelz;
        loadnid=k1*(nelx+1)*(nely+1)+i1*(nely+1)+(nely+1-j1);
        loaddof=3*loadnid(:)-1;
        iif=[0 0 nelx nelx]; jf=[0 0 0 0]; kf=[0 nelz 0 nelz];
        fixednid=kf*(nelx+1)*(nely+1)+iif*(nely+1)+(nely+1-jf);
        fixeddof=[3*fixednid(:);3*fixednid(:)-1;3*fixednid(:)-2];
end
%% PREPARE FINITE ELEMENT ANALYSIS
ndof = 3*(nelx+1)*(nely+1)*(nelz+1);
F = 0.1*sparse(loaddof,1,-1,ndof,1);
U = zeros(ndof,1);
freedofs = setdiff(1:ndof,fixeddof);
KE = lk_H8(nu);
nodegrd = reshape(1:(nely+1)*(nelx+1),nely+1,nelx+1);
nodeids = reshape(nodegrd(1:end-1,1:end-1),nely*nelx,1);
nodeidz = 0:(nely+1)*(nelx+1):(nelz-1)*(nely+1)*(nelx+1);
nodeids = repmat(nodeids,size(nodeidz))+repmat(nodeidz,size(nodeids));
edofVec = 3*nodeids(:)+1;
edofMat = repmat(edofVec,1,24)+ ...
    repmat([0 1 2 3*nely + [3 4 5 0 1 2] -3 -2 -1 ...
    3*(nely+1)*(nelx+1)+[0 1 2 3*nely + [3 4 5 0 1 2] -3 -2 -1]],nele,1);
iK = reshape(kron(edofMat,ones(24,1))',24*24*nele,1);
jK = reshape(kron(edofMat,ones(1,24))',24*24*nele,1);
%% FILTER
[H,Hs]=HHs(nelx,nely,nelz,rmin);
[Hn,Hns]=HnHns(nelx,nely,nelz);
%% INITIALIZE ITERATION
vx = repmat(vol,[nely,nelx,nelz]); vxPhys = vx; loop = 0; change = 1;
%% INITIALIZE MMA OPTIMIZER
m = 1; n = nele; xmin = zeros(n,1); xmax = ones(n,1); xold1 = vx(:);
xold2 = vx(:); low = ones(n,1); upp = ones(n,1); a0 = 1;
a = zeros(m,1); c_MMA = 10000*ones(m,1); d = zeros(m,1);
%% START ITERATION
vxPrint=AMfilter3D(vxPhys, baseplate, TopMax);
while (change > tolx || tol>0.001) && loop < maxloop
    %% UPDATE ITERATION
    loop = loop+1;
    %% FE-ANALYSIS
    sK = reshape(KE(:)*(vxPrint(:)'*E0+(1-vxPrint(:))'*(Emin*E0)),24*24*nele,1);
    K = sparse(iK(:),jK(:),sK(:)); K = (K+K')/2;
    U(freedofs,:)=K(freedofs, freedofs)\F(freedofs,:);
    %% OBJECTIVE FUNCTION AND SENSITIVITY ANALYSIS
    ce = reshape(sum((U(edofMat)*KE).*U(edofMat),2),[nely,nelx,nelz]);
    c(loop)= sum(sum(sum((vxPrint.*E0+(1-vxPrint).*(Emin*E0)).*ce)));
    dc =-((1-vxPrint)*Emin+vxPrint).*E0.*ce;
    dv = ones(nely,nelx,nelz);
    %% FILTERING AND MODIFICATION OF SENSITIVITIES
    dc(:) = H*(dc(:)./Hs);
    dv(:) = H*(dv(:)./Hs);
    [vxPrint, dc, dv]=AMfilter3D(vxPhys,baseplate, TopMax, dc,dv);
    %% METHOD OF MOVING ASYMPTOTES
    xval  = vx(:);
    f0val = Obj_Scal*c;
    df0dx = Obj_Scal*dc(:);
    fval  = sum(vxPrint(:))/(vol*nele) - 1;
    dfdx  = dv(:)' / (vol*nele);
    [vxmma, ~, ~, ~, ~, ~, ~, ~, ~, low,upp] = ...
        mmasub(m, n, loop, xval, xmin, xmax, xold1, xold2, ...
        f0val,df0dx,fval,dfdx,low,upp,a0,a,c_MMA,d);
    %% Update MMA Variables
    vxnew     = reshape(vxmma,nely,nelx,nelz);
    vxPhys(:) = (H*vxnew(:))./Hs;
    xold2    = xold1(:);
    xold1    = vx(:);
    vxPrint=AMfilter3D(vxPhys,baseplate, TopMax);
    xn = reshape((Hn*vxPrint(:)./Hns),nely+1,nelx+1,nelz+1);
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH FUNCTION/ HEAVISIDE STEP FUNCTION
    xg = interp3(nodex,nodey,nodez,xn,fnx,fny,fnz,'linear');
    l1 =0; l2 = 1;
    while (l2-l1) > 1.0e-5
        ls = (l1+l2)/2.0;
        %% Heaviside smooth function
        xgnew = max(0.001,(tanh(beta*ls)+tanh(beta*(xg-ls)))/(tanh(beta*ls)+tanh(beta*(1-ls))));
        if sum(sum(sum(xgnew)))/((ngrid*nelx+1)*(ngrid*nely+1)*(ngrid*nelz+1)) - sum(vxPrint(:))/(nele) > 0
            l1 = ls;
        else
            l2 = ls;
        end
    end
    %% CONVERTING TO ELEMENTS
    vxPrint(:) = 0;
    Terr = 0;
    Tm=[];
    for nk = 1:nelz
        for ni = 1:nelx
            for nj = 1:nely
                ne = (nk-1)*nelx*nely+(ni-1)*nely+nj;
                for nk1 = ngrid*(nk-1)+1:ngrid*nk+1
                    for ni1 = ngrid*(ni-1)+1:ngrid*ni+1
                        for nj1 = ngrid*(nj-1)+1:ngrid*nj+1
                            Tm=[Tm;xgnew(nj1,ni1,nk1)];
                            vxPrint(ne) = vxPrint(ne)+xgnew(nj1,ni1,nk1);
                        end
                    end
                end
                if min(Tm)>0.001 && max(Tm)<1
                    Terr=Terr+1;
                end
                Tm=[];
            end
        end
    end
    vxPrint = vxPrint/((ngrid+1)^3);
    %% Topology Error
    tol = Terr/nele;
    %% CHECK CONVERGENCE
    change = sum(abs(vxnew(:)-vx(:)))/(vol*nele);
    vx=vxnew;
    %% PRINT RESULTS
    fprintf('It.:%5i Obj.:%11.3f Vol.:%7.3f ch.:%7.5f Topo.:%7.5f\n',loop,c(loop),mean(vxPrint(:)),change,tol);
    figure(1)
    clf;
    top = xg-ls;
    display_3Dsmooth(xg,top,loop);
    figure(2)
    clf;
    plot(c,'.-');
    set (gca,'xgrid','on','ygrid','on')
    hold on;
    plot(loop,c(loop),'r*','LineWidth',2,'MarkerSize',8)
    xlabel('Iteration')
    ylabel('Compliance (J)', 'rotation', 90 )
    pause(1e-6);
    %% UPDATE% HEAVISIDE REGULARIZATION PARAMETER
    beta=beta+ER; fprintf('Parameter beta increased to %g.\n',beta);
end
end
%% === GENERATE ELEMENT STIFFNESS MATRIX ===
function [KE] = lk_H8(nu)
A = [32 6 -8 6 -6 4 3 -6 -10 3 -3 -3 -4 -8;
    -48 0 0 -24 24 0 0 0 12 -12 0 12 12 12];
k = 1/144*A'*[1; nu];
K1 = [k(1) k(2) k(2) k(3) k(5) k(5);
    k(2) k(1) k(2) k(4) k(6) k(7);
    k(2) k(2) k(1) k(4) k(7) k(6);
    k(3) k(4) k(4) k(1) k(8) k(8);
    k(5) k(6) k(7) k(8) k(1) k(2);
    k(5) k(7) k(6) k(8) k(2) k(1)];
K2 = [k(9)  k(8)  k(12) k(6)  k(4)  k(7);
    k(8)  k(9)  k(12) k(5)  k(3)  k(5);
    k(10) k(10) k(13) k(7)  k(4)  k(6);
    k(6)  k(5)  k(11) k(9)  k(2)  k(10);
    k(4)  k(3)  k(5)  k(2)  k(9)  k(12)
    k(11) k(4)  k(6)  k(12) k(10) k(13)];
K3 = [k(6)  k(7)  k(4)  k(9)  k(12) k(8);
    k(7)  k(6)  k(4)  k(10) k(13) k(10);
    k(5)  k(5)  k(3)  k(8)  k(12) k(9);
    k(9)  k(10) k(2)  k(6)  k(11) k(5);
    k(12) k(13) k(10) k(11) k(6)  k(4);
    k(2)  k(12) k(9)  k(4)  k(5)  k(3)];
K4 = [k(14) k(11) k(11) k(13) k(10) k(10);
    k(11) k(14) k(11) k(12) k(9)  k(8);
    k(11) k(11) k(14) k(12) k(8)  k(9);
    k(13) k(12) k(12) k(14) k(7)  k(7);
    k(10) k(9)  k(8)  k(7)  k(14) k(11);
    k(10) k(8)  k(9)  k(7)  k(11) k(14)];
K5 = [k(1) k(2)  k(8)  k(3) k(5)  k(4);
    k(2) k(1)  k(8)  k(4) k(6)  k(11);
    k(8) k(8)  k(1)  k(5) k(11) k(6);
    k(3) k(4)  k(5)  k(1) k(8)  k(2);
    k(5) k(6)  k(11) k(8) k(1)  k(8);
    k(4) k(11) k(6)  k(2) k(8)  k(1)];
K6 = [k(14) k(11) k(7)  k(13) k(10) k(12);
    k(11) k(14) k(7)  k(12) k(9)  k(2);
    k(7)  k(7)  k(14) k(10) k(2)  k(9);
    k(13) k(12) k(10) k(14) k(7)  k(11);
    k(10) k(9)  k(2)  k(7)  k(14) k(7);
    k(12) k(2)  k(9)  k(11) k(7)  k(14)];
KE = 1/((nu+1)*(1-2*nu))*...
    [ K1  K2  K3  K4;
    K2'  K5  K6  K3';
    K3' K6  K5' K2';
    K4  K3  K2  K1'];
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada    %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% The basic structure of the code is based on the 3D code from:           %
% "An efficient 3D topology optimization code written in Matlab",         %
% K Liu and A Tovar, SMO 2014, doi:10.1007/s00158-014-1107-x              %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
